<?php
// includes/functions.php

// Database configuration constants
define('DB_HOST', 'localhost:3306');
define('DB_NAME', 'blog');
define('DB_USER', 'root'); // Change as needed
define('DB_PASS', ''); // Change as needed

// Upload configuration
define('UPLOAD_DIR', __DIR__ . '/../uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_TYPES', ['jpg', 'jpeg', 'png', 'gif', 'webp']);

/**
 * Generate a URL-friendly slug from a string
 * 
 * @param string $text The string to convert to slug
 * @return string The generated slug
 */
function generateSlug($text) {
    // Convert to lowercase
    $slug = strtolower($text);
    
    // Replace spaces with hyphens
    $slug = preg_replace('/\s+/', '-', $slug);
    
    // Remove all non-alphanumeric characters except hyphens
    $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
    
    // Remove multiple consecutive hyphens
    $slug = preg_replace('/-+/', '-', $slug);
    
    // Trim hyphens from beginning and end
    $slug = trim($slug, '-');
    
    // If empty, generate random slug
    if (empty($slug)) {
        $slug = 'post-' . uniqid();
    }
    
    return $slug;
}

/**
 * Check if a slug already exists in the database
 * 
 * @param PDO $pdo Database connection
 * @param string $slug The slug to check
 * @param int $exclude_id Post ID to exclude (for updates)
 * @return bool True if slug exists, false otherwise
 */
function slugExists($pdo, $slug, $exclude_id = null) {
    $query = "SELECT COUNT(*) FROM blog_posts WHERE slug = :slug";
    $params = [':slug' => $slug];
    
    if ($exclude_id) {
        $query .= " AND id != :exclude_id";
        $params[':exclude_id'] = $exclude_id;
    }
    
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    
    return $stmt->fetchColumn() > 0;
}

/**
 * Generate a unique slug
 * 
 * @param PDO $pdo Database connection
 * @param string $title The post title
 * @param int $exclude_id Post ID to exclude (for updates)
 * @return string Unique slug
 */
function generateUniqueSlug($pdo, $title, $exclude_id = null) {
    $base_slug = generateSlug($title);
    $slug = $base_slug;
    $counter = 1;
    
    // Keep adding numbers until we find a unique slug
    while (slugExists($pdo, $slug, $exclude_id)) {
        $slug = $base_slug . '-' . $counter;
        $counter++;
    }
    
    return $slug;
}

/**
 * Generate a URL-friendly slug from a string (alternative method)
 * 
 * @param string $text The string to convert to slug
 * @return string The generated slug
 */
function slugify($text) {
    $text = preg_replace('~[^\pL\d]+~u', '-', $text);
    $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
    $text = preg_replace('~[^-\w]+~', '', $text);
    $text = trim($text, '-');
    $text = preg_replace('~-+~', '-', $text);
    $text = strtolower($text);
    
    if (empty($text)) {
        return 'n-a';
    }
    
    return $text;
}

/**
 * Upload an image file
 * 
 * @param array $file The $_FILES array element
 * @return array Result with success/error and filename
 */
function uploadImage($file) {
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['error' => 'Upload failed'];
    }
    
    if ($file['size'] > MAX_FILE_SIZE) {
        return ['error' => 'File too large'];
    }
    
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($ext, ALLOWED_TYPES)) {
        return ['error' => 'Invalid file type'];
    }
    
    $filename = uniqid() . '_' . time() . '.' . $ext;
    $destination = UPLOAD_DIR . $filename;
    
    // Create upload directory if it doesn't exist
    if (!file_exists(UPLOAD_DIR)) {
        mkdir(UPLOAD_DIR, 0777, true);
    }
    
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        return ['success' => true, 'filename' => $filename];
    }
    
    return ['error' => 'Upload failed'];
}

/**
 * Get blog posts
 * 
 * @param int $limit Number of posts to retrieve (0 for all)
 * @param string $status Post status filter
 * @return array Array of posts
 */
function getPosts($limit = 10, $status = 'published') {
    global $pdo;
    
    $sql = "SELECT * FROM blog_posts WHERE status = ? ORDER BY published_at DESC";
    if ($limit > 0) {
        $sql .= " LIMIT ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$status, $limit]);
    } else {
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$status]);
    }
    
    return $stmt->fetchAll();
}

/**
 * Get a single post by ID
 * 
 * @param int $id Post ID
 * @return array|null Post data or null if not found
 */
function getPostById($id) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT * FROM blog_posts WHERE id = ?");
    $stmt->execute([$id]);
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get a single post by slug
 * 
 * @param string $slug Post slug
 * @return array|null Post data or null if not found
 */
function getPostBySlug($slug) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT * FROM blog_posts WHERE slug = ?");
    $stmt->execute([$slug]);
    
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Format date for display
 * 
 * @param string $date Date string
 * @param string $format Date format
 * @return string Formatted date
 */
function formatDate($date, $format = 'F j, Y') {
    return date($format, strtotime($date));
}

/**
 * Get excerpt from content
 * 
 * @param string $content Full post content
 * @param int $length Excerpt length
 * @return string Excerpt
 */
function getExcerpt($content, $length = 150) {
    $text = strip_tags($content);
    if (strlen($text) > $length) {
        $text = substr($text, 0, $length);
        $text = substr($text, 0, strrpos($text, ' ')) . '...';
    }
    return $text;
}

/**
 * Increase post view count
 * 
 * @param int $id Post ID
 */
function incrementViewCount($id) {
    global $pdo;
    
    $stmt = $pdo->prepare("UPDATE blog_posts SET views = views + 1 WHERE id = ?");
    $stmt->execute([$id]);
}

/**
 * Sanitize output
 * 
 * @param string $string String to sanitize
 * @return string Sanitized string
 */
function sanitize($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Redirect to a URL
 * 
 * @param string $url URL to redirect to
 * @param int $status HTTP status code
 */
function redirect($url, $status = 302) {
    header("Location: $url", true, $status);
    exit();
}

/**
 * Set flash message
 * 
 * @param string $message Message text
 * @param string $type Message type (success, error, warning, info)
 */
function setFlashMessage($message, $type = 'info') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
}

/**
 * Get and clear flash message
 * 
 * @return array|null Message array or null if no message
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = [
            'text' => $_SESSION['flash_message'],
            'type' => $_SESSION['flash_type'] ?? 'info'
        ];
        
        unset($_SESSION['flash_message'], $_SESSION['flash_type']);
        return $message;
    }
    
    return null;
}

/**
 * Check if user is admin
 * 
 * @return bool True if admin is logged in
 */
function isAdmin() {
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

/**
 * Require admin login
 */
function requireAdmin() {
    if (!isAdmin()) {
        redirect('login.php');
    }
}

/**
 * Generate pagination
 * 
 * @param int $total_items Total number of items
 * @param int $current_page Current page number
 * @param int $per_page Items per page
 * @param string $base_url Base URL for pagination links
 * @return string HTML pagination
 */
function generatePagination($total_items, $current_page, $per_page, $base_url) {
    $total_pages = ceil($total_items / $per_page);
    
    if ($total_pages <= 1) {
        return '';
    }
    
    $html = '<div class="pagination">';
    
    // Previous link
    if ($current_page > 1) {
        $html .= '<a href="' . $base_url . '&page=' . ($current_page - 1) . '" class="page-link">Previous</a>';
    }
    
    // Page numbers
    for ($i = 1; $i <= $total_pages; $i++) {
        $active = $i == $current_page ? ' active' : '';
        $html .= '<a href="' . $base_url . '&page=' . $i . '" class="page-link' . $active . '">' . $i . '</a>';
    }
    
    // Next link
    if ($current_page < $total_pages) {
        $html .= '<a href="' . $base_url . '&page=' . ($current_page + 1) . '" class="page-link">Next</a>';
    }
    
    $html .= '</div>';
    
    return $html;
}
?>