<?php
// elementor-snippet.php
// Blog Posts Grid with Popup Viewer

// Database connection
$host = 'localhost:3306';
$dbname = 'blog';
$username = 'root'; // Change to your database username
$password = ''; // Change to your database password

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Get all published posts
$stmt = $pdo->prepare("SELECT * FROM blog_posts WHERE status = 'published' ORDER BY published_at DESC");
$stmt->execute();
$posts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Function to get excerpt from content
function getExcerpt($content, $length = 150) {
    $text = strip_tags($content);
    if (strlen($text) > $length) {
        $text = substr($text, 0, $length);
        $text = substr($text, 0, strrpos($text, ' ')) . '...';
    }
    return $text;
}

// Function to format date
function formatDate($date) {
    return date('F j, Y', strtotime($date));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Blog Posts</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* Reset and Base Styles */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            line-height: 1.6;
            color: #333;
            background: #f8f9fa;
        }

        .blog-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 40px 20px;
        }

        /* Header */
        .blog-header {
            text-align: center;
            margin-bottom: 60px;
        }

        .blog-header h1 {
            font-size: 3rem;
            font-weight: 700;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 15px;
        }

        .blog-header p {
            color: #666;
            font-size: 1.1rem;
            max-width: 600px;
            margin: 0 auto;
        }

        /* Blog Grid */
        .blog-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 30px;
            margin-bottom: 60px;
        }

        @media (max-width: 768px) {
            .blog-grid {
                grid-template-columns: 1fr;
            }
        }

        /* Blog Card */
        .blog-card {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
            transition: all 0.4s ease;
            height: 100%;
            display: flex;
            flex-direction: column;
        }

        .blog-card:hover {
            transform: translateY(-10px);
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.12);
        }

        /* Image Container */
        .blog-image {
            height: 220px;
            overflow: hidden;
            position: relative;
        }

        .blog-image img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            transition: transform 0.5s ease;
        }

        .blog-card:hover .blog-image img {
            transform: scale(1.05);
        }

        .image-placeholder {
            height: 100%;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 3rem;
        }

        /* Card Content */
        .blog-content {
            padding: 25px;
            flex-grow: 1;
            display: flex;
            flex-direction: column;
        }

        .blog-meta {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 15px;
            font-size: 0.9rem;
            color: #666;
        }

        .blog-date {
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .blog-views {
            display: flex;
            align-items: center;
            gap: 5px;
        }

        .blog-title {
            font-size: 1.4rem;
            font-weight: 700;
            margin-bottom: 15px;
            line-height: 1.3;
            color: #222;
        }

        .blog-excerpt {
            color: #555;
            margin-bottom: 20px;
            line-height: 1.6;
            flex-grow: 1;
        }

        /* Read More Button */
        .read-more-btn {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 12px 24px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-decoration: none;
            border-radius: 50px;
            font-weight: 600;
            font-size: 0.95rem;
            border: none;
            cursor: pointer;
            transition: all 0.3s ease;
            width: fit-content;
            margin-top: auto;
        }

        .read-more-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.3);
        }

        /* Popup Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            z-index: 1000;
            overflow-y: auto;
            padding: 20px;
        }

        .modal-content {
            background: white;
            max-width: 900px;
            margin: 40px auto;
            border-radius: 20px;
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.2);
            overflow: hidden;
            animation: modalSlideIn 0.4s ease;
        }

        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .modal-header {
            padding: 30px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            position: relative;
        }

        .modal-title {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 10px;
            line-height: 1.3;
        }

        .modal-meta {
            display: flex;
            gap: 20px;
            font-size: 0.95rem;
            opacity: 0.9;
        }

        .close-btn {
            position: absolute;
            top: 25px;
            right: 25px;
            background: rgba(255, 255, 255, 0.2);
            border: none;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            color: white;
            font-size: 1.2rem;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }

        .close-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: rotate(90deg);
        }

        .modal-body {
            padding: 40px;
            max-height: 70vh;
            overflow-y: auto;
        }

        .modal-image {
            width: 100%;
            height: 400px;
            object-fit: cover;
            border-radius: 12px;
            margin-bottom: 30px;
        }

        .modal-body-content {
            font-size: 1.1rem;
            line-height: 1.8;
            color: #333;
        }

        /* Preserve all formatting from Quill editor */
        .modal-body-content p {
            margin-bottom: 1.5em;
            color: #444;
        }

        .modal-body-content h1,
        .modal-body-content h2,
        .modal-body-content h3,
        .modal-body-content h4,
        .modal-body-content h5,
        .modal-body-content h6 {
            margin: 1.5em 0 0.8em 0;
            color: #222;
            font-weight: 700;
            line-height: 1.3;
        }

        .modal-body-content h1 { font-size: 2em; }
        .modal-body-content h2 { font-size: 1.75em; }
        .modal-body-content h3 { font-size: 1.5em; }
        .modal-body-content h4 { font-size: 1.25em; }

        .modal-body-content strong {
            font-weight: 700;
            color: #222;
        }

        .modal-body-content em {
            font-style: italic;
        }

        .modal-body-content u {
            text-decoration: underline;
        }

        .modal-body-content a {
            color: #667eea;
            text-decoration: none;
            font-weight: 500;
            border-bottom: 1px solid rgba(102, 126, 234, 0.3);
            transition: all 0.3s ease;
        }

        .modal-body-content a:hover {
            color: #764ba2;
            border-bottom-color: #764ba2;
        }

        .modal-body-content ul,
        .modal-body-content ol {
            margin: 1.5em 0;
            padding-left: 2em;
        }

        .modal-body-content li {
            margin-bottom: 0.8em;
            color: #444;
        }

        .modal-body-content blockquote {
            border-left: 4px solid #667eea;
            padding-left: 20px;
            margin: 2em 0;
            font-style: italic;
            color: #555;
        }

        .modal-body-content img {
            max-width: 100%;
            height: auto;
            border-radius: 8px;
            margin: 2em 0;
        }

        .modal-body-content .ql-align-center {
            text-align: center;
        }

        .modal-body-content .ql-align-right {
            text-align: right;
        }

        .modal-body-content .ql-align-left {
            text-align: left;
        }

        /* Loading Animation */
        .loading {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }

        .loading i {
            font-size: 3rem;
            margin-bottom: 20px;
            color: #667eea;
            animation: spin 2s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* No Posts Message */
        .no-posts {
            text-align: center;
            padding: 80px 20px;
            color: #666;
        }

        .no-posts i {
            font-size: 4rem;
            margin-bottom: 20px;
            color: #ddd;
        }

        .no-posts h3 {
            font-size: 1.5rem;
            margin-bottom: 10px;
            color: #888;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .blog-header h1 {
                font-size: 2.2rem;
            }

            .blog-grid {
                gap: 20px;
            }

            .blog-content {
                padding: 20px;
            }

            .modal-content {
                margin: 20px auto;
            }

            .modal-header {
                padding: 20px;
            }

            .modal-title {
                font-size: 1.6rem;
                padding-right: 40px;
            }

            .modal-body {
                padding: 25px;
                max-height: 80vh;
            }

            .modal-image {
                height: 250px;
            }

            .modal-body-content {
                font-size: 1rem;
            }

            .close-btn {
                top: 15px;
                right: 15px;
                width: 35px;
                height: 35px;
            }
        }

        @media (max-width: 480px) {
            .blog-container {
                padding: 20px 15px;
            }

            .blog-header {
                margin-bottom: 40px;
            }

            .blog-header h1 {
                font-size: 1.8rem;
            }

            .blog-title {
                font-size: 1.2rem;
            }

            .read-more-btn {
                width: 100%;
                justify-content: center;
            }

            .modal-body {
                padding: 20px;
            }

            .modal-image {
                height: 200px;
            }
        }
    </style>
</head>
<body>
    <div class="blog-container">
        <!-- Header -->
        <div class="blog-header">
            <h1>Latest Blog Posts</h1>
            <p>Insights, news, and updates from our team</p>
        </div>

        <!-- Blog Grid -->
        <?php if (empty($posts)): ?>
            <div class="no-posts">
                <i class="fas fa-newspaper"></i>
                <h3>No blog posts yet</h3>
                <p>Check back soon for updates!</p>
            </div>
        <?php else: ?>
            <div class="blog-grid">
                <?php foreach ($posts as $post): ?>
                    <div class="blog-card" data-post-id="<?= $post['id'] ?>">
                        <!-- Image -->
                        <div class="blog-image">
                            <?php if ($post['featured_image']): ?>
                                <img src="uploads/<?= htmlspecialchars($post['featured_image']) ?>" 
                                     alt="<?= htmlspecialchars($post['title']) ?>"
                                     onerror="this.style.display='none'; this.parentElement.innerHTML='<div class=\"image-placeholder\"><i class=\"fas fa-newspaper\"></i></div>';">
                            <?php else: ?>
                                <div class="image-placeholder">
                                    <i class="fas fa-newspaper"></i>
                                </div>
                            <?php endif; ?>
                        </div>

                        <!-- Content -->
                        <div class="blog-content">
                            <!-- Meta Info -->
                            <div class="blog-meta">
                                <span class="blog-date">
                                    <i class="far fa-calendar"></i>
                                    <?= formatDate($post['published_at'] ?: $post['created_at']) ?>
                                </span>
                                <span class="blog-views">
                                    <i class="far fa-eye"></i>
                                    <?= $post['views'] ?> views
                                </span>
                            </div>

                            <!-- Title -->
                            <h3 class="blog-title"><?= htmlspecialchars($post['title']) ?></h3>

                            <!-- Excerpt -->
                            <p class="blog-excerpt">
                                <?= htmlspecialchars(getExcerpt($post['excerpt'] ?: $post['content'])) ?>
                            </p>

                            <!-- Read More Button -->
                            <button class="read-more-btn" onclick="openBlogModal(<?= $post['id'] ?>)">
                                <i class="fas fa-book-open"></i>
                                Read More
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Popup Modal -->
    <div id="blogModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <button class="close-btn" onclick="closeBlogModal()">
                    <i class="fas fa-times"></i>
                </button>
                <h2 id="modalTitle" class="modal-title">Loading...</h2>
                <div id="modalMeta" class="modal-meta">
                    <span><i class="far fa-calendar"></i> <span id="modalDate">...</span></span>
                    <span><i class="far fa-eye"></i> <span id="modalViews">0</span> views</span>
                </div>
            </div>
            <div class="modal-body">
                <div id="modalLoading" class="loading" style="display: block;">
                    <i class="fas fa-spinner"></i>
                    <p>Loading article...</p>
                </div>
                <div id="modalContent" style="display: none;">
                    <img id="modalImage" class="modal-image" alt="Featured image">
                    <div id="modalBodyContent" class="modal-body-content">
                        <!-- Blog content will be inserted here -->
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Open blog modal
        function openBlogModal(postId) {
            const modal = document.getElementById('blogModal');
            const modalContent = document.getElementById('modalContent');
            const modalLoading = document.getElementById('modalLoading');
            
            // Show modal with loading state
            modal.style.display = 'block';
            modalContent.style.display = 'none';
            modalLoading.style.display = 'block';
            
            // Prevent body scroll
            document.body.style.overflow = 'hidden';
            
            // Fetch post data via AJAX
            fetch(`get-post.php?id=${postId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Update modal with post data
                        document.getElementById('modalTitle').textContent = data.title;
                        document.getElementById('modalDate').textContent = formatDate(data.published_at || data.created_at);
                        document.getElementById('modalViews').textContent = data.views;
                        
                        // Update image
                        const modalImage = document.getElementById('modalImage');
                        if (data.featured_image) {
                            modalImage.src = 'uploads/' + data.featured_image;
                            modalImage.style.display = 'block';
                            modalImage.onerror = function() {
                                this.style.display = 'none';
                            };
                        } else {
                            modalImage.style.display = 'none';
                        }
                        
                        // Update content - IMPORTANT: don't escape HTML here!
                        document.getElementById('modalBodyContent').innerHTML = data.content;
                        
                        // Hide loading, show content
                        modalLoading.style.display = 'none';
                        modalContent.style.display = 'block';
                        
                        // Update view count
                        updateViewCount(postId);
                    } else {
                        modalLoading.innerHTML = '<i class="fas fa-exclamation-triangle"></i><p>Error loading post. Please try again.</p>';
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    modalLoading.innerHTML = '<i class="fas fa-exclamation-triangle"></i><p>Error loading post. Please try again.</p>';
                });
        }

        // Close blog modal
        function closeBlogModal() {
            const modal = document.getElementById('blogModal');
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('blogModal');
            if (event.target === modal) {
                closeBlogModal();
            }
        }

        // Close modal with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeBlogModal();
            }
        });

        // Update view count
        function updateViewCount(postId) {
            // Send AJAX request to update view count
            fetch(`update-views.php?id=${postId}`, { method: 'POST' })
                .catch(error => console.error('Error updating view count:', error));
        }

        // Format date function
        function formatDate(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            });
        }
    </script>

    <?php
    // Create the required helper files if they don't exist
    ?>
</body>
</html>

<?php
// Create helper files automatically
$helper_dir = dirname(__FILE__);
$get_post_file = $helper_dir . '/get-post.php';
$update_views_file = $helper_dir . '/update-views.php';

// Create get-post.php if it doesn't exist
if (!file_exists($get_post_file)) {
    $get_post_content = '<?php
// get-post.php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");

// Database connection
$host = "localhost:3306";
$dbname = "blog";
$username = "root";
$password = "";

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    echo json_encode(["success" => false, "error" => "Database connection failed"]);
    exit;
}

if (!isset($_GET["id"]) || !is_numeric($_GET["id"])) {
    echo json_encode(["success" => false, "error" => "Invalid post ID"]);
    exit;
}

$post_id = (int)$_GET["id"];

$stmt = $pdo->prepare("SELECT * FROM blog_posts WHERE id = ? AND status = ?published?");
$stmt->execute([$post_id, "published"]);
$post = $stmt->fetch(PDO::FETCH_ASSOC);

if ($post) {
    echo json_encode([
        "success" => true,
        "id" => $post["id"],
        "title" => $post["title"],
        "content" => $post["content"],
        "excerpt" => $post["excerpt"],
        "featured_image" => $post["featured_image"],
        "published_at" => $post["published_at"],
        "created_at" => $post["created_at"],
        "views" => $post["views"]
    ]);
} else {
    echo json_encode(["success" => false, "error" => "Post not found"]);
}
?>';
    file_put_contents($get_post_file, $get_post_content);
}

// Create update-views.php if it doesn't exist
if (!file_exists($update_views_file)) {
    $update_views_content = '<?php
// update-views.php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");

if ($_SERVER["REQUEST_METHOD"] === "POST") {
    // Database connection
    $host = "localhost:3306";
    $dbname = "blog";
    $username = "root";
    $password = "";
    
    try {
        $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } catch(PDOException $e) {
        exit;
    }
    
    if (isset($_GET["id"]) && is_numeric($_GET["id"])) {
        $post_id = (int)$_GET["id"];
        $stmt = $pdo->prepare("UPDATE blog_posts SET views = views + 1 WHERE id = ?");
        $stmt->execute([$post_id]);
    }
}
?>';
    file_put_contents($update_views_file, $update_views_content);
}
?>