<?php
session_start();
// Security headers
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

require_once __DIR__ . "/../includes/auth.php";
require_login();
require_once __DIR__ . "/../includes/config.php";
require_once __DIR__ . "/../includes/functions.php";

// Get current admin info
$admin_id = $_SESSION['admin_id'];
$stmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
$stmt->execute([$admin_id]);
$admin = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$admin) {
    $_SESSION['message'] = "Admin account not found.";
    $_SESSION['message_type'] = 'error';
    header("Location: logout.php");
    exit();
}

// Initialize variables
$errors = [];
$success_message = '';
$current_email = $admin['email'];
$current_username = $admin['username'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get form data
    $email = trim($_POST['email'] ?? '');
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // Validate email
    if (empty($email)) {
        $errors[] = "Email is required.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Please enter a valid email address.";
    } elseif (strlen($email) > 100) {
        $errors[] = "Email must be less than 100 characters.";
    }
    
    // Check if email is being changed
    $email_changed = ($email !== $current_email);
    
    // Check if password is being changed
    $password_changed = !empty($new_password) || !empty($confirm_password);
    
    // If password is being changed
    if ($password_changed) {
        if (empty($current_password)) {
            $errors[] = "Current password is required to change password.";
        } elseif (!password_verify($current_password, $admin['password'])) {
            $errors[] = "Current password is incorrect.";
        }
        
        if (empty($new_password)) {
            $errors[] = "New password is required.";
        } elseif (strlen($new_password) < 8) {
            $errors[] = "New password must be at least 8 characters.";
        }
        
        if ($new_password !== $confirm_password) {
            $errors[] = "New passwords do not match.";
        }
    }
    
    // If email is being changed, check if it already exists
    if ($email_changed) {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM admin_users WHERE email = ? AND id != ?");
        $stmt->execute([$email, $admin_id]);
        if ($stmt->fetchColumn() > 0) {
            $errors[] = "This email is already registered to another account.";
        }
    }
    
    // If no errors, update the profile
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Prepare update query
            $update_data = [];
            $update_fields = [];
            
            // Add email if changed
            if ($email_changed) {
                $update_fields[] = "email = :email";
                $update_data[':email'] = $email;
            }
            
            // Add password if changed
            if ($password_changed) {
                $update_fields[] = "password = :password";
                $update_data[':password'] = password_hash($new_password, PASSWORD_DEFAULT);
            }
            
            // Only update if something changed
            if (!empty($update_fields)) {
                $update_fields[] = "updated_at = NOW()";
                
                $update_query = "UPDATE admin_users SET " . implode(", ", $update_fields) . " WHERE id = :id";
                $update_data[':id'] = $admin_id;
                
                $stmt = $pdo->prepare($update_query);
                $stmt->execute($update_data);
                
                $pdo->commit();
                
                // Update session email if changed
                if ($email_changed) {
                    $_SESSION['admin_email'] = $email;
                    $current_email = $email;
                }
                
                $success_message = "Profile updated successfully!";
                
                // Refresh admin data
                $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE id = ?");
                $stmt->execute([$admin_id]);
                $admin = $stmt->fetch(PDO::FETCH_ASSOC);
                
            } else {
                $errors[] = "No changes were made.";
            }
            
        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Profile - Blog Admin</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }
        
        .admin-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        /* Header Styles */
        .admin-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            padding: 25px 30px;
            margin-bottom: 30px;
            color: white;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .admin-header h1 {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .admin-info {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            font-size: 14px;
            opacity: 0.9;
            margin-top: 10px;
        }
        
        .admin-info span {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .header-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            margin-top: 20px;
        }
        
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            border: none;
            cursor: pointer;
            font-size: 14px;
        }
        
        .btn-primary {
            background: #4CAF50;
            color: white;
        }
        
        .btn-primary:hover {
            background: #45a049;
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: #2196F3;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #1976D2;
        }
        
        .btn-light {
            background: white;
            color: #333;
            border: 1px solid #ddd;
        }
        
        .btn-light:hover {
            background: #f8f9fa;
        }
        
        .btn-danger {
            background: #f44336;
            color: white;
        }
        
        .btn-danger:hover {
            background: #d32f2f;
        }
        
        /* Main Content */
        .profile-container {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
        }
        
        @media (max-width: 768px) {
            .profile-container {
                grid-template-columns: 1fr;
            }
        }
        
        /* Form Section */
        .form-section {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.08);
        }
        
        .form-section h2 {
            font-size: 20px;
            font-weight: 600;
            margin-bottom: 25px;
            color: #333;
            padding-bottom: 15px;
            border-bottom: 2px solid #f0f0f0;
        }
        
        /* Error Messages */
        .errors {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            list-style-type: none;
        }
        
        .errors li {
            margin: 5px 0;
            padding-left: 20px;
            position: relative;
        }
        
        .errors li:before {
            content: "⚠️";
            position: absolute;
            left: 0;
        }
        
        /* Success Message */
        .success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        /* Form Groups */
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #444;
            font-size: 15px;
        }
        
        .form-group .required:after {
            content: " *";
            color: #f44336;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 15px;
            transition: border-color 0.3s ease;
            font-family: inherit;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .form-control:disabled {
            background: #f8f9fa;
            cursor: not-allowed;
        }
        
        .password-strength {
            height: 5px;
            background: #eee;
            border-radius: 3px;
            margin-top: 8px;
            overflow: hidden;
        }
        
        .strength-meter {
            height: 100%;
            width: 0;
            transition: width 0.3s ease, background-color 0.3s ease;
            border-radius: 3px;
        }
        
        .strength-weak { background: #f44336; }
        .strength-medium { background: #ff9800; }
        .strength-strong { background: #4CAF50; }
        
        .password-hints {
            font-size: 13px;
            color: #666;
            margin-top: 8px;
        }
        
        .password-hints ul {
            padding-left: 20px;
            margin-top: 5px;
        }
        
        .password-hints li {
            margin-bottom: 3px;
        }
        
        .password-hints .valid {
            color: #4CAF50;
        }
        
        .password-hints .invalid {
            color: #f44336;
        }
        
        /* Info Box */
        .info-box {
            background: #e8f4fd;
            border: 1px solid #b6d4fe;
            border-radius: 8px;
            padding: 20px;
            margin-top: 30px;
        }
        
        .info-box h3 {
            color: #0d6efd;
            margin-bottom: 10px;
            font-size: 16px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .info-box p {
            color: #666;
            font-size: 14px;
            line-height: 1.5;
        }
        
        /* Account Stats */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
            margin-top: 20px;
        }
        
        .stat-card {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            border: 1px solid #e9ecef;
        }
        
        .stat-card .number {
            font-size: 24px;
            font-weight: 700;
            color: #667eea;
            margin-bottom: 5px;
        }
        
        .stat-card .label {
            font-size: 13px;
            color: #666;
        }
        
        /* Form Actions */
        .form-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            justify-content: flex-end;
            padding-top: 25px;
            margin-top: 25px;
            border-top: 1px solid #eee;
        }
        
        .form-actions .btn {
            min-width: 120px;
            justify-content: center;
        }
        
        /* Loading Indicator */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Password Toggle */
        .password-toggle {
            position: relative;
        }
        
        .toggle-password {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: #666;
            cursor: pointer;
            font-size: 18px;
        }
        
        /* Mobile Optimizations */
        @media (max-width: 768px) {
            .admin-container {
                padding: 15px;
            }
            
            .admin-header {
                padding: 20px;
            }
            
            .admin-header h1 {
                font-size: 24px;
            }
            
            .header-actions {
                flex-direction: column;
            }
            
            .header-actions .btn {
                width: 100%;
                justify-content: center;
            }
            
            .form-section {
                padding: 20px;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .form-actions .btn {
                width: 100%;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 480px) {
            .admin-container {
                padding: 10px;
            }
            
            .form-section {
                padding: 15px;
            }
            
            .btn {
                padding: 12px 15px;
                font-size: 13px;
            }
        }
        
        /* Last Login Info */
        .last-login {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 20px;
            font-size: 14px;
            color: #666;
        }
        
        .last-login strong {
            color: #333;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Header -->
        <div class="admin-header">
            <h1>👤 Admin Profile</h1>
            <div class="admin-info">
                <span>👋 Welcome, <strong><?= htmlspecialchars($admin['username']) ?></strong></span>
                <span>📧 Email: <?= htmlspecialchars($admin['email']) ?></span>
                <span>📅 Member since: <?= date('F j, Y', strtotime($admin['created_at'])) ?></span>
            </div>
            <div class="header-actions">
                <a href="index.php" class="btn btn-light">
                    ← Back to Dashboard
                </a>
                <a href="posts.php" class="btn btn-light">
                    📄 Manage Posts
                </a>
            </div>
        </div>
        
        <!-- Error Messages -->
        <?php if (!empty($errors)): ?>
            <ul class="errors">
                <?php foreach ($errors as $error): ?>
                    <li><?= htmlspecialchars($error) ?></li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>
        
        <!-- Success Message -->
        <?php if ($success_message): ?>
            <div class="success">
                <span>✅ <?= htmlspecialchars($success_message) ?></span>
                <button onclick="this.parentElement.style.display='none'">✕</button>
            </div>
        <?php endif; ?>
        
        <div class="profile-container">
            <!-- Profile Update Form -->
            <div class="form-section">
                <h2>Update Profile Information</h2>
                
                <form method="POST" action="" id="profileForm">
                    <!-- Email -->
                    <div class="form-group">
                        <label for="email" class="required">Email Address</label>
                        <input type="email" 
                               id="email" 
                               name="email" 
                               class="form-control" 
                               value="<?= htmlspecialchars($current_email) ?>" 
                               required
                               maxlength="100"
                               placeholder="your.email@example.com">
                    </div>
                    
                    <!-- Username (Read-only) -->
                    <div class="form-group">
                        <label for="username">Username</label>
                        <input type="text" 
                               id="username" 
                               class="form-control" 
                               value="<?= htmlspecialchars($current_username) ?>" 
                               disabled
                               placeholder="Username cannot be changed">
                        <small style="color: #666; font-size: 13px; margin-top: 5px; display: block;">
                            Username cannot be changed for security reasons
                        </small>
                    </div>
                    
                    <!-- Password Section -->
                    <div class="form-group">
                        <label>Change Password (Optional)</label>
                        <p style="color: #666; font-size: 14px; margin-bottom: 15px;">
                            Leave blank if you don't want to change your password
                        </p>
                        
                        <!-- Current Password -->
                        <div class="form-group" style="margin-bottom: 20px;">
                            <label for="current_password">Current Password</label>
                            <div class="password-toggle">
                                <input type="password" 
                                       id="current_password" 
                                       name="current_password" 
                                       class="form-control" 
                                       autocomplete="current-password"
                                       placeholder="Enter your current password">
                                <button type="button" class="toggle-password" data-target="current_password">
                                    👁️
                                </button>
                            </div>
                        </div>
                        
                        <!-- New Password -->
                        <div class="form-group" style="margin-bottom: 20px;">
                            <label for="new_password">New Password</label>
                            <div class="password-toggle">
                                <input type="password" 
                                       id="new_password" 
                                       name="new_password" 
                                       class="form-control" 
                                       autocomplete="new-password"
                                       placeholder="Enter new password (min. 8 characters)">
                                <button type="button" class="toggle-password" data-target="new_password">
                                    👁️
                                </button>
                            </div>
                            <div class="password-strength">
                                <div class="strength-meter" id="password-strength-meter"></div>
                            </div>
                            <div class="password-hints" id="password-hints">
                                <strong>Password must contain:</strong>
                                <ul>
                                    <li id="hint-length" class="invalid">At least 8 characters</li>
                                    <li id="hint-uppercase" class="invalid">One uppercase letter</li>
                                    <li id="hint-lowercase" class="invalid">One lowercase letter</li>
                                    <li id="hint-number" class="invalid">One number</li>
                                    <li id="hint-special" class="invalid">One special character</li>
                                </ul>
                            </div>
                        </div>
                        
                        <!-- Confirm Password -->
                        <div class="form-group">
                            <label for="confirm_password">Confirm New Password</label>
                            <div class="password-toggle">
                                <input type="password" 
                                       id="confirm_password" 
                                       name="confirm_password" 
                                       class="form-control" 
                                       autocomplete="new-password"
                                       placeholder="Confirm your new password">
                                <button type="button" class="toggle-password" data-target="confirm_password">
                                    👁️
                                </button>
                            </div>
                            <div id="password-match" style="font-size: 13px; margin-top: 5px;"></div>
                        </div>
                    </div>
                    
                    <!-- Form Actions -->
                    <div class="form-actions">
                        <button type="reset" class="btn btn-light">
                            Reset Form
                        </button>
                        <button type="submit" class="btn btn-primary">
                            💾 Update Profile
                        </button>
                    </div>
                </form>
                
                <!-- Security Info -->
                <div class="info-box">
                    <h3><i class="fas fa-shield-alt"></i> Security Tips</h3>
                    <p>• Use a strong, unique password that you don't use elsewhere</p>
                    <p>• Never share your password with anyone</p>
                    <p>• Consider using a password manager</p>
                    <p>• Log out after each session, especially on shared computers</p>
                </div>
            </div>
            
            <!-- Account Information -->
            <div class="form-section">
                <h2>Account Information</h2>
                
                <!-- Account Stats -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="number"><?= date('M j', strtotime($admin['created_at'])) ?></div>
                        <div class="label">Account Created</div>
                    </div>
                    
                    <?php
                    // Get post count for this admin (if you track who created posts)
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM blog_posts");
                    $stmt->execute();
                    $total_posts = $stmt->fetchColumn();
                    ?>
                    
                    <div class="stat-card">
                        <div class="number"><?= $total_posts ?></div>
                        <div class="label">Total Posts</div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="number">
                            <?php
                            $stmt = $pdo->prepare("SELECT COUNT(*) FROM blog_posts WHERE status = 'published'");
                            $stmt->execute();
                            $published_posts = $stmt->fetchColumn();
                            echo $published_posts;
                            ?>
                        </div>
                        <div class="label">Published Posts</div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="number">
                            <?php
                            $stmt = $pdo->prepare("SELECT COUNT(*) FROM blog_posts WHERE status = 'draft'");
                            $stmt->execute();
                            $draft_posts = $stmt->fetchColumn();
                            echo $draft_posts;
                            ?>
                        </div>
                        <div class="label">Draft Posts</div>
                    </div>
                </div>
                
                <!-- Last Login Info -->
                <div class="last-login">
                    <p><strong>Last Login:</strong> 
                        <?php 
                        if (isset($_SESSION['last_login'])) {
                            echo date('F j, Y \a\t g:i A', $_SESSION['last_login']);
                        } else {
                            echo 'First login';
                        }
                        ?>
                    </p>
                    <p><strong>Current Session:</strong> Started at <?= date('g:i A', $_SESSION['login_time'] ?? time()) ?></p>
                </div>
                
                <!-- Danger Zone -->
                <div style="margin-top: 40px; padding-top: 20px; border-top: 2px solid #f0f0f0;">
                    <h3 style="color: #f44336; margin-bottom: 15px;">⚠️ Danger Zone</h3>
                    
                    <div style="background: #fff5f5; border: 1px solid #f8d7da; border-radius: 8px; padding: 20px;">
                        <h4 style="color: #721c24; margin-bottom: 10px;">Log Out All Sessions</h4>
                        <p style="color: #666; margin-bottom: 15px; font-size: 14px;">
                            This will log you out from all devices where you're currently logged in.
                            You'll need to log in again.
                        </p>
                        <form method="POST" action="logout.php" style="display: inline;">
                            <input type="hidden" name="logout_all" value="1">
                            <button type="submit" class="btn btn-danger" 
                                    onclick="return confirm('Are you sure you want to log out from all devices?')">
                                🚪 Log Out Everywhere
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Password toggle visibility
            document.querySelectorAll('.toggle-password').forEach(button => {
                button.addEventListener('click', function() {
                    const targetId = this.getAttribute('data-target');
                    const input = document.getElementById(targetId);
                    const type = input.getAttribute('type') === 'password' ? 'text' : 'password';
                    input.setAttribute('type', type);
                    
                    // Update eye icon
                    this.textContent = type === 'password' ? '👁️' : '🙈';
                });
            });
            
            // Password strength checker
            const passwordInput = document.getElementById('new_password');
            const strengthMeter = document.getElementById('password-strength-meter');
            const passwordHints = document.getElementById('password-hints');
            
            if (passwordInput) {
                passwordInput.addEventListener('input', function() {
                    const password = this.value;
                    let strength = 0;
                    
                    // Check length
                    const hasLength = password.length >= 8;
                    document.getElementById('hint-length').className = hasLength ? 'valid' : 'invalid';
                    if (hasLength) strength += 20;
                    
                    // Check uppercase
                    const hasUppercase = /[A-Z]/.test(password);
                    document.getElementById('hint-uppercase').className = hasUppercase ? 'valid' : 'invalid';
                    if (hasUppercase) strength += 20;
                    
                    // Check lowercase
                    const hasLowercase = /[a-z]/.test(password);
                    document.getElementById('hint-lowercase').className = hasLowercase ? 'valid' : 'invalid';
                    if (hasLowercase) strength += 20;
                    
                    // Check numbers
                    const hasNumber = /[0-9]/.test(password);
                    document.getElementById('hint-number').className = hasNumber ? 'valid' : 'invalid';
                    if (hasNumber) strength += 20;
                    
                    // Check special characters
                    const hasSpecial = /[^A-Za-z0-9]/.test(password);
                    document.getElementById('hint-special').className = hasSpecial ? 'valid' : 'invalid';
                    if (hasSpecial) strength += 20;
                    
                    // Update strength meter
                    strengthMeter.style.width = strength + '%';
                    
                    // Update color based on strength
                    if (strength < 40) {
                        strengthMeter.className = 'strength-meter strength-weak';
                    } else if (strength < 80) {
                        strengthMeter.className = 'strength-meter strength-medium';
                    } else {
                        strengthMeter.className = 'strength-meter strength-strong';
                    }
                });
            }
            
            // Password match checker
            const newPasswordInput = document.getElementById('new_password');
            const confirmPasswordInput = document.getElementById('confirm_password');
            const matchIndicator = document.getElementById('password-match');
            
            function checkPasswordMatch() {
                const newPassword = newPasswordInput.value;
                const confirmPassword = confirmPasswordInput.value;
                
                if (newPassword === '' && confirmPassword === '') {
                    matchIndicator.textContent = '';
                    matchIndicator.style.color = '';
                } else if (newPassword === confirmPassword) {
                    matchIndicator.textContent = '✓ Passwords match';
                    matchIndicator.style.color = '#4CAF50';
                } else {
                    matchIndicator.textContent = '✗ Passwords do not match';
                    matchIndicator.style.color = '#f44336';
                }
            }
            
            if (newPasswordInput && confirmPasswordInput) {
                newPasswordInput.addEventListener('input', checkPasswordMatch);
                confirmPasswordInput.addEventListener('input', checkPasswordMatch);
            }
            
            // Form submission loading state
            const form = document.getElementById('profileForm');
            if (form) {
                form.addEventListener('submit', function(e) {
                    const submitButton = this.querySelector('button[type="submit"]');
                    if (submitButton) {
                        const originalText = submitButton.innerHTML;
                        submitButton.innerHTML = '<span class="loading"></span> Updating...';
                        submitButton.disabled = true;
                        
                        setTimeout(() => {
                            submitButton.innerHTML = originalText;
                            submitButton.disabled = false;
                        }, 3000);
                    }
                });
            }
            
            // Auto-hide messages after 5 seconds
            setTimeout(() => {
                const successMessage = document.querySelector('.success');
                if (successMessage) {
                    successMessage.style.opacity = '0';
                    setTimeout(() => successMessage.style.display = 'none', 300);
                }
                
                const errorMessages = document.querySelector('.errors');
                if (errorMessages) {
                    errorMessages.style.opacity = '0';
                    setTimeout(() => errorMessages.style.display = 'none', 300);
                }
            }, 5000);
            
            // Warn before leaving if form has changes
            let formChanged = false;
            const formInputs = form.querySelectorAll('input, textarea, select');
            
            formInputs.forEach(input => {
                input.addEventListener('input', () => formChanged = true);
                input.addEventListener('change', () => formChanged = true);
            });
            
            window.addEventListener('beforeunload', (e) => {
                if (formChanged) {
                    e.preventDefault();
                    e.returnValue = 'You have unsaved changes. Are you sure you want to leave?';
                }
            });
            
            form.addEventListener('submit', () => formChanged = false);
        });
        
        // Generate strong password suggestion
        function generatePassword() {
            const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*()_+-=[]{}|;:,.<>?';
            let password = '';
            
            // Ensure at least one of each required type
            password += 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'[Math.floor(Math.random() * 26)];
            password += 'abcdefghijklmnopqrstuvwxyz'[Math.floor(Math.random() * 26)];
            password += '0123456789'[Math.floor(Math.random() * 10)];
            password += '!@#$%^&*()_+-=[]{}|;:,.<>?'[Math.floor(Math.random() * 28)];
            
            // Add remaining random characters
            for (let i = 4; i < 12; i++) {
                password += chars[Math.floor(Math.random() * chars.length)];
            }
            
            // Shuffle the password
            password = password.split('').sort(() => 0.5 - Math.random()).join('');
            
            // Set the password fields
            document.getElementById('new_password').value = password;
            document.getElementById('confirm_password').value = password;
            
            // Trigger events to update UI
            document.getElementById('new_password').dispatchEvent(new Event('input'));
            document.getElementById('confirm_password').dispatchEvent(new Event('input'));
        }
        
        // Add generate password button (optional)
        const passwordSection = document.querySelector('.form-group label[for="new_password"]');
        if (passwordSection) {
            const generateBtn = document.createElement('button');
            generateBtn.type = 'button';
            generateBtn.className = 'btn btn-light';
            generateBtn.style.marginLeft = '10px';
            generateBtn.style.fontSize = '12px';
            generateBtn.style.padding = '5px 10px';
            generateBtn.innerHTML = '🎲 Generate Strong Password';
            generateBtn.onclick = generatePassword;
            passwordSection.appendChild(generateBtn);
        }
    </script>
</body>
</html>