<?php
session_start();
// Security headers
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

require_once __DIR__ . "/../includes/auth.php";
require_login();
require_once __DIR__ . "/../includes/config.php";

// Handle post deletion
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $post_id = (int)$_GET['delete'];
    
    // Get image path before deleting
    $stmt = $pdo->prepare("SELECT featured_image FROM blog_posts WHERE id = ?");
    $stmt->execute([$post_id]);
    $post = $stmt->fetch();
    
    if ($post) {
        // Delete post from database
        $stmt = $pdo->prepare("DELETE FROM blog_posts WHERE id = ?");
        $stmt->execute([$post_id]);
        
        // Delete associated image file if exists
        if ($post['featured_image'] && file_exists(__DIR__ . "/../uploads/" . $post['featured_image'])) {
            unlink(__DIR__ . "/../uploads/" . $post['featured_image']);
        }
        
        $_SESSION['message'] = "Post deleted successfully!";
        $_SESSION['message_type'] = "success";
    }
    
    header("Location: posts.php");
    exit();
}

// Handle bulk actions
if (isset($_POST['bulk_action']) && isset($_POST['selected_posts'])) {
    $selected_posts = $_POST['selected_posts'];
    
    if ($_POST['bulk_action'] == 'delete') {
        // Get all images to delete
        $placeholders = rtrim(str_repeat('?,', count($selected_posts)), ',');
        $stmt = $pdo->prepare("SELECT featured_image FROM blog_posts WHERE id IN ($placeholders)");
        $stmt->execute($selected_posts);
        $images = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        // Delete image files
        foreach ($images as $image) {
            if ($image && file_exists(__DIR__ . "/../uploads/" . $image)) {
                unlink(__DIR__ . "/../uploads/" . $image);
            }
        }
        
        // Delete posts from database
        $stmt = $pdo->prepare("DELETE FROM blog_posts WHERE id IN ($placeholders)");
        $stmt->execute($selected_posts);
        
        $_SESSION['message'] = count($selected_posts) . " posts deleted successfully!";
        $_SESSION['message_type'] = "success";
    } elseif ($_POST['bulk_action'] == 'publish') {
        $placeholders = rtrim(str_repeat('?,', count($selected_posts)), ',');
        $stmt = $pdo->prepare("UPDATE blog_posts SET status = 'published', published_at = NOW() WHERE id IN ($placeholders)");
        $stmt->execute($selected_posts);
        
        $_SESSION['message'] = count($selected_posts) . " posts published successfully!";
        $_SESSION['message_type'] = "success";
    } elseif ($_POST['bulk_action'] == 'draft') {
        $placeholders = rtrim(str_repeat('?,', count($selected_posts)), ',');
        $stmt = $pdo->prepare("UPDATE blog_posts SET status = 'draft' WHERE id IN ($placeholders)");
        $stmt->execute($selected_posts);
        
        $_SESSION['message'] = count($selected_posts) . " posts moved to draft!";
        $_SESSION['message_type'] = "info";
    }
    
    header("Location: posts.php");
    exit();
}

// Pagination setup
$per_page = 10;
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$offset = ($page - 1) * $per_page;

// Filter by status
$filter_status = isset($_GET['status']) && in_array($_GET['status'], ['published', 'draft']) ? $_GET['status'] : null;
$filter_clause = $filter_status ? "WHERE status = :status" : "";

// Get total posts for pagination
$total_query = "SELECT COUNT(*) FROM blog_posts" . ($filter_status ? " WHERE status = :status" : "");
$stmt = $pdo->prepare($total_query);
if ($filter_status) {
    $stmt->bindParam(':status', $filter_status);
}
$stmt->execute();
$total_posts = $stmt->fetchColumn();
$total_pages = ceil($total_posts / $per_page);

// Fetch posts with pagination
$query = "SELECT * FROM blog_posts $filter_clause ORDER BY created_at DESC LIMIT :limit OFFSET :offset";
$stmt = $pdo->prepare($query);
$stmt->bindValue(':limit', $per_page, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
if ($filter_status) {
    $stmt->bindParam(':status', $filter_status);
}
$stmt->execute();
$posts = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Posts - Blog Admin</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }
        
        .admin-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }
        
        /* Header Styles */
        .admin-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            padding: 25px 30px;
            margin-bottom: 30px;
            color: white;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .admin-header h1 {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 15px;
        }
        
        .header-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            align-items: center;
        }
        
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            border: none;
            cursor: pointer;
            font-size: 14px;
        }
        
        .btn-primary {
            background: #4CAF50;
            color: white;
        }
        
        .btn-primary:hover {
            background: #45a049;
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: #2196F3;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #1976D2;
            transform: translateY(-2px);
        }
        
        .btn-danger {
            background: #f44336;
            color: white;
        }
        
        .btn-danger:hover {
            background: #d32f2f;
            transform: translateY(-2px);
        }
        
        .btn-light {
            background: white;
            color: #333;
            border: 1px solid #ddd;
        }
        
        .btn-light:hover {
            background: #f8f9fa;
        }
        
        /* Message Styles */
        .message {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            animation: slideDown 0.3s ease;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .message.info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Filter and Bulk Actions */
        .action-bar {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 25px;
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            align-items: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .filter-buttons {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        .filter-btn {
            padding: 8px 16px;
            border-radius: 6px;
            text-decoration: none;
            font-size: 13px;
            transition: all 0.3s ease;
            background: #f8f9fa;
            color: #495057;
            border: 1px solid #dee2e6;
        }
        
        .filter-btn.active {
            background: #007bff;
            color: white;
            border-color: #007bff;
        }
        
        .filter-btn:hover:not(.active) {
            background: #e9ecef;
        }
        
        .bulk-actions {
            display: flex;
            gap: 10px;
            align-items: center;
            margin-left: auto;
        }
        
        .bulk-select {
            padding: 8px 12px;
            border-radius: 6px;
            border: 1px solid #ddd;
            background: white;
        }
        
        /* Posts Grid */
        .posts-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 25px;
        }
        
        @media (max-width: 768px) {
            .posts-grid {
                grid-template-columns: 1fr;
            }
        }
        
        .post-card {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.08);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .post-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.12);
        }
        
        .post-image {
            width: 100%;
            height: 200px;
            object-fit: cover;
            border-bottom: 3px solid #f0f0f0;
        }
        
        .no-image {
            height: 200px;
            background: linear-gradient(45deg, #f0f0f0, #e0e0e0);
            display: flex;
            align-items: center;
            justify-content: center;
            color: #888;
            font-size: 14px;
            border-bottom: 3px solid #f0f0f0;
        }
        
        .post-content {
            padding: 20px;
        }
        
        .post-title {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 10px;
            color: #333;
            line-height: 1.4;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .post-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            margin-bottom: 15px;
            font-size: 12px;
            color: #666;
        }
        
        .status-badge {
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-published {
            background: #d4edda;
            color: #155724;
        }
        
        .status-draft {
            background: #fff3cd;
            color: #856404;
        }
        
        .post-excerpt {
            color: #666;
            font-size: 14px;
            line-height: 1.5;
            margin-bottom: 20px;
            display: -webkit-box;
            -webkit-line-clamp: 3;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
        
        .post-actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid #eee;
        }
        
        .post-actions .btn {
            flex: 1;
            justify-content: center;
            padding: 8px 12px;
            font-size: 13px;
        }
        
        /* Pagination */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
            margin-top: 40px;
            flex-wrap: wrap;
        }
        
        .page-link {
            padding: 8px 16px;
            border-radius: 6px;
            text-decoration: none;
            background: white;
            color: #007bff;
            border: 1px solid #dee2e6;
            transition: all 0.3s ease;
            font-size: 14px;
        }
        
        .page-link:hover {
            background: #f8f9fa;
        }
        
        .page-link.active {
            background: #007bff;
            color: white;
            border-color: #007bff;
        }
        
        .page-link.disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        /* Checkbox styling */
        .post-select {
            position: absolute;
            top: 15px;
            left: 15px;
            width: 20px;
            height: 20px;
            z-index: 1;
        }
        
        .post-card {
            position: relative;
        }
        
        /* Mobile Optimizations */
        @media (max-width: 768px) {
            .admin-container {
                padding: 15px;
            }
            
            .admin-header {
                padding: 20px;
            }
            
            .admin-header h1 {
                font-size: 24px;
            }
            
            .header-actions {
                flex-direction: column;
                align-items: stretch;
            }
            
            .action-bar {
                flex-direction: column;
                align-items: stretch;
            }
            
            .bulk-actions {
                margin-left: 0;
                flex-wrap: wrap;
            }
            
            .post-meta {
                flex-direction: column;
                gap: 5px;
            }
            
            .post-actions {
                flex-direction: column;
            }
            
            .btn {
                width: 100%;
                justify-content: center;
            }
            
            .bulk-select {
                width: 100%;
            }
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            background: white;
            border-radius: 12px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.05);
        }
        
        .empty-state-icon {
            font-size: 60px;
            margin-bottom: 20px;
            color: #ddd;
        }
        
        .empty-state h3 {
            font-size: 22px;
            color: #666;
            margin-bottom: 10px;
        }
        
        .empty-state p {
            color: #888;
            margin-bottom: 20px;
        }
        
        /* Loading skeleton for images */
        .image-skeleton {
            width: 100%;
            height: 200px;
            background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%);
            background-size: 200% 100%;
            animation: loading 1.5s infinite;
        }
        
        @keyframes loading {
            0% { background-position: 200% 0; }
            100% { background-position: -200% 0; }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Header -->
        <div class="admin-header">
            <h1>📝 Manage Blog Posts</h1>
            <div class="header-actions">
                <a href="index.php" class="btn btn-light">
                    ← Back to Dashboard
                </a>
                <a href="new-post.php" class="btn btn-primary">
                    ＋ Add New Post
                </a>
            </div>
        </div>
        
        <!-- Messages -->
        <?php if (isset($_SESSION['message'])): ?>
            <div class="message <?= $_SESSION['message_type'] ?? 'info' ?>">
                <span><?= htmlspecialchars($_SESSION['message']) ?></span>
                <button onclick="this.parentElement.style.display='none'">✕</button>
            </div>
            <?php unset($_SESSION['message'], $_SESSION['message_type']); ?>
        <?php endif; ?>
        
        <!-- Action Bar -->
        <form method="POST" action="" id="bulkForm">
            <div class="action-bar">
                <div class="filter-buttons">
                    <a href="?status=all" class="filter-btn <?= !$filter_status ? 'active' : '' ?>">All Posts (<?= $total_posts ?>)</a>
                    <a href="?status=published" class="filter-btn <?= $filter_status == 'published' ? 'active' : '' ?>">Published</a>
                    <a href="?status=draft" class="filter-btn <?= $filter_status == 'draft' ? 'active' : '' ?>">Drafts</a>
                </div>
                
                <div class="bulk-actions">
                    <select name="bulk_action" class="bulk-select" onchange="if(this.value) document.getElementById('bulkForm').submit();">
                        <option value="">Bulk Actions</option>
                        <option value="publish">Publish Selected</option>
                        <option value="draft">Move to Draft</option>
                        <option value="delete">Delete Selected</option>
                    </select>
                    <button type="button" class="btn btn-secondary" onclick="selectAllPosts()">Select All</button>
                </div>
            </div>
            
            <!-- Posts Grid -->
            <div class="posts-grid">
                <?php if (empty($posts)): ?>
                    <div class="empty-state">
                        <div class="empty-state-icon">📄</div>
                        <h3>No posts found</h3>
                        <p><?= $filter_status ? "No {$filter_status} posts available." : "You haven't created any posts yet." ?></p>
                        <a href="new-post.php" class="btn btn-primary">Create Your First Post</a>
                    </div>
                <?php else: ?>
                    <?php foreach ($posts as $post): ?>
                        <div class="post-card">
                            <input type="checkbox" name="selected_posts[]" value="<?= $post['id'] ?>" class="post-select">
                            
                            <?php if ($post['featured_image']): ?>
                                <img src="../uploads/<?= htmlspecialchars($post['featured_image']) ?>" 
                                     alt="<?= htmlspecialchars($post['title']) ?>" 
                                     class="post-image"
                                     onerror="this.style.display='none'; this.parentElement.innerHTML='<div class=\'no-image\'><span>Image not found</span></div>';">
                            <?php else: ?>
                                <div class="no-image">
                                    <span>No featured image</span>
                                </div>
                            <?php endif; ?>
                            
                            <div class="post-content">
                                <h3 class="post-title"><?= htmlspecialchars($post['title']) ?></h3>
                                
                                <div class="post-meta">
                                    <span>📅 <?= date('M d, Y', strtotime($post['created_at'])) ?></span>
                                    <span>👁 <?= $post['views'] ?> views</span>
                                    <span class="status-badge status-<?= $post['status'] ?>">
                                        <?= ucfirst($post['status']) ?>
                                    </span>
                                    <?php if ($post['published_at']): ?>
                                        <span>🚀 Published: <?= date('M d, Y', strtotime($post['published_at'])) ?></span>
                                    <?php endif; ?>
                                </div>
                                
                                <?php if ($post['excerpt']): ?>
                                    <p class="post-excerpt"><?= htmlspecialchars($post['excerpt']) ?></p>
                                <?php endif; ?>
                                
                                <div class="post-actions">
                                    <a href="edit-post.php?id=<?= $post['id'] ?>" class="btn btn-secondary">✏️ Edit</a>
                                    <a href="?delete=<?= $post['id'] ?>" 
                                       class="btn btn-danger"
                                       onclick="return confirm('Are you sure you want to delete this post? This action cannot be undone.')">🗑️ Delete</a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </form>
        
        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
            <div class="pagination">
                <?php if ($page > 1): ?>
                    <a href="?page=<?= $page - 1 ?><?= $filter_status ? "&status={$filter_status}" : '' ?>" class="page-link">← Previous</a>
                <?php else: ?>
                    <span class="page-link disabled">← Previous</span>
                <?php endif; ?>
                
                <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                    <a href="?page=<?= $i ?><?= $filter_status ? "&status={$filter_status}" : '' ?>" 
                       class="page-link <?= $i == $page ? 'active' : '' ?>">
                        <?= $i ?>
                    </a>
                <?php endfor; ?>
                
                <?php if ($page < $total_pages): ?>
                    <a href="?page=<?= $page + 1 ?><?= $filter_status ? "&status={$filter_status}" : '' ?>" class="page-link">Next →</a>
                <?php else: ?>
                    <span class="page-link disabled">Next →</span>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        // Select all posts functionality
        function selectAllPosts() {
            const checkboxes = document.querySelectorAll('.post-select');
            const allChecked = Array.from(checkboxes).every(cb => cb.checked);
            
            checkboxes.forEach(cb => {
                cb.checked = !allChecked;
            });
            
            // Update select all button text
            const btn = document.querySelector('.btn-secondary');
            btn.textContent = allChecked ? 'Select All' : 'Deselect All';
        }
        
        // Image error handling
        document.addEventListener('DOMContentLoaded', function() {
            // Auto-hide messages after 5 seconds
            const messages = document.querySelectorAll('.message');
            messages.forEach(message => {
                setTimeout(() => {
                    message.style.opacity = '0';
                    setTimeout(() => message.style.display = 'none', 300);
                }, 5000);
            });
            
            // Bulk form confirmation
            const bulkForm = document.getElementById('bulkForm');
            if (bulkForm) {
                bulkForm.addEventListener('submit', function(e) {
                    const selectedPosts = this.querySelectorAll('input[name="selected_posts[]"]:checked');
                    const action = this.querySelector('select[name="bulk_action"]').value;
                    
                    if (selectedPosts.length === 0) {
                        e.preventDefault();
                        alert('Please select at least one post.');
                        return;
                    }
                    
                    if (action === 'delete') {
                        if (!confirm(`Are you sure you want to delete ${selectedPosts.length} post(s)? This action cannot be undone.`)) {
                            e.preventDefault();
                        }
                    }
                });
            }
        });
    </script>
</body>
</html>