<?php
session_start();
// Security headers
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

require_once __DIR__ . "/../includes/auth.php";
require_login();
require_once __DIR__ . "/../includes/config.php";
require_once __DIR__ . "/../includes/functions.php";

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get form data
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');
    $excerpt = trim($_POST['excerpt'] ?? '');
    $status = $_POST['status'] ?? 'draft';
    
    // Validate required fields
    $errors = [];
    
    if (empty($title)) {
        $errors[] = "Title is required.";
    } elseif (strlen($title) > 200) {
        $errors[] = "Title must be less than 200 characters.";
    }
    
    if (empty($content)) {
        $errors[] = "Content is required.";
    }
    
    if (strlen($excerpt) > 500) {
        $errors[] = "Excerpt must be less than 500 characters.";
    }
    
    // Handle image upload
    $featured_image = null;
    
    if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] === UPLOAD_ERR_OK) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $max_size = 5 * 1024 * 1024; // 5MB
        
        $file_type = $_FILES['featured_image']['type'];
        $file_size = $_FILES['featured_image']['size'];
        $file_tmp = $_FILES['featured_image']['tmp_name'];
        $file_name = $_FILES['featured_image']['name'];
        
        // Check file type
        if (!in_array($file_type, $allowed_types)) {
            $errors[] = "Only JPG, PNG, GIF, and WebP images are allowed.";
        }
        
        // Check file size
        if ($file_size > $max_size) {
            $errors[] = "Image size must be less than 5MB.";
        }
        
        // If no errors, process the upload
        if (empty($errors)) {
            // Generate unique filename
            $file_extension = pathinfo($file_name, PATHINFO_EXTENSION);
            $new_filename = uniqid() . '_' . time() . '.' . $file_extension;
            $upload_path = __DIR__ . "/../uploads/" . $new_filename;
            
            // Create uploads directory if it doesn't exist
            if (!file_exists(__DIR__ . "/../uploads")) {
                mkdir(__DIR__ . "/../uploads", 0777, true);
            }
            
            // Move uploaded file
            if (move_uploaded_file($file_tmp, $upload_path)) {
                $featured_image = $new_filename;
            } else {
                $errors[] = "Failed to upload image. Please try again.";
            }
        }
    } elseif (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] !== UPLOAD_ERR_NO_FILE) {
        // Handle upload errors
        $upload_errors = [
            UPLOAD_ERR_INI_SIZE => 'The uploaded file exceeds the upload_max_filesize directive in php.ini.',
            UPLOAD_ERR_FORM_SIZE => 'The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form.',
            UPLOAD_ERR_PARTIAL => 'The uploaded file was only partially uploaded.',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder.',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk.',
            UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload.',
        ];
        
        $error_code = $_FILES['featured_image']['error'];
        $errors[] = $upload_errors[$error_code] ?? 'Unknown upload error.';
    }
    
    // If no errors, insert the post
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Generate unique slug from title using the new function
            $slug = generateUniqueSlug($pdo, $title);
            
            // Set published_at timestamp if publishing
            $published_at = ($status === 'published') ? date('Y-m-d H:i:s') : null;
            
            // Insert post
            $insert_query = "INSERT INTO blog_posts 
                            (title, slug, content, excerpt, featured_image, status, published_at, created_at, updated_at) 
                            VALUES 
                            (:title, :slug, :content, :excerpt, :featured_image, :status, :published_at, NOW(), NOW())";
            
            $stmt = $pdo->prepare($insert_query);
            $stmt->execute([
                ':title' => $title,
                ':slug' => $slug,
                ':content' => $content,
                ':excerpt' => $excerpt,
                ':featured_image' => $featured_image,
                ':status' => $status,
                ':published_at' => $published_at
            ]);
            
            $post_id = $pdo->lastInsertId();
            
            $pdo->commit();
            
            $_SESSION['message'] = "Post created successfully!";
            $_SESSION['message_type'] = 'success';
            header("Location: edit-post.php?id=" . $post_id);
            exit();
            
        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create New Post - Blog Admin</title>
    
    <!-- Quill Rich Text Editor -->
    <link href="https://cdn.quilljs.com/1.3.7/quill.snow.css" rel="stylesheet">
    <script src="https://cdn.quilljs.com/1.3.7/quill.min.js"></script>
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }
        
        .admin-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        /* Header Styles */
        .admin-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            padding: 20px 25px;
            margin-bottom: 25px;
            color: white;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
        }
        
        .admin-header h1 {
            font-size: 24px;
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .header-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 15px;
        }
        
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            border: none;
            cursor: pointer;
            font-size: 14px;
        }
        
        .btn-primary {
            background: #4CAF50;
            color: white;
        }
        
        .btn-primary:hover {
            background: #45a049;
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: #2196F3;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #1976D2;
        }
        
        .btn-light {
            background: white;
            color: #333;
            border: 1px solid #ddd;
        }
        
        .btn-light:hover {
            background: #f8f9fa;
        }
        
        /* Form Container */
        .form-container {
            background: white;
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 25px;
        }
        
        /* Error Messages */
        .errors {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            list-style-type: none;
        }
        
        .errors li {
            margin: 5px 0;
            padding-left: 20px;
            position: relative;
        }
        
        .errors li:before {
            content: "⚠️";
            position: absolute;
            left: 0;
        }
        
        /* Success Message */
        .success-message {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        /* Form Groups */
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #444;
            font-size: 15px;
        }
        
        .form-group .required:after {
            content: " *";
            color: #f44336;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 15px;
            transition: border-color 0.3s ease;
            font-family: inherit;
        }
        
        .form-control:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        /* Character Counters */
        .char-counter {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
            text-align: right;
        }
        
        /* Rich Text Editor */
        .editor-wrapper {
            border: 1px solid #ddd;
            border-radius: 8px;
            overflow: hidden;
            margin-bottom: 10px;
        }
        
        #editor {
            height: 400px;
            font-size: 16px;
            line-height: 1.6;
        }
        
        /* Quill editor customization */
        .ql-toolbar {
            border: none !important;
            border-bottom: 1px solid #ddd !important;
            background: #f8f9fa;
        }
        
        .ql-container {
            border: none !important;
            font-family: inherit;
        }
        
        .ql-editor {
            min-height: 350px;
            padding: 20px;
        }
        
        .ql-editor.ql-blank::before {
            color: #999;
            font-style: normal;
            left: 20px;
        }
        
        /* Image Upload */
        .image-upload {
            border: 2px dashed #ddd;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            background: #fafafa;
        }
        
        .image-preview {
            max-width: 100%;
            height: 150px;
            object-fit: cover;
            border-radius: 8px;
            margin: 15px auto;
            display: none;
        }
        
        .no-image {
            background: linear-gradient(45deg, #f0f0f0, #e0e0e0);
            height: 100px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #888;
            margin: 10px 0;
        }
        
        .file-input-container {
            position: relative;
            margin: 15px 0;
        }
        
        .file-input {
            position: absolute;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            opacity: 0;
            cursor: pointer;
        }
        
        .file-input-label {
            display: inline-block;
            padding: 10px 20px;
            background: #667eea;
            color: white;
            border-radius: 8px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s ease;
        }
        
        .file-input-label:hover {
            background: #5a6fd8;
        }
        
        .file-info {
            font-size: 13px;
            color: #666;
            margin-top: 8px;
        }
        
        /* Status Options */
        .status-options {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .status-option {
            display: flex;
            align-items: center;
            gap: 8px;
            padding: 10px 15px;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #dee2e6;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .status-option:hover {
            background: #e9ecef;
        }
        
        .status-option.active {
            background: #667eea;
            border-color: #667eea;
            color: white;
        }
        
        .status-option input[type="radio"] {
            display: none;
        }
        
        /* Form Actions */
        .form-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            justify-content: flex-end;
            padding-top: 25px;
            margin-top: 25px;
            border-top: 1px solid #eee;
        }
        
        .form-actions .btn {
            min-width: 120px;
            justify-content: center;
        }
        
        /* Loading Indicator */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Mobile Optimizations */
        @media (max-width: 768px) {
            .admin-container {
                padding: 15px;
            }
            
            .admin-header {
                padding: 15px 20px;
            }
            
            .admin-header h1 {
                font-size: 20px;
            }
            
            .header-actions {
                flex-direction: column;
            }
            
            .header-actions .btn {
                width: 100%;
                justify-content: center;
            }
            
            .form-container {
                padding: 20px;
            }
            
            .form-actions {
                flex-direction: column;
            }
            
            .form-actions .btn {
                width: 100%;
            }
            
            .status-options {
                flex-direction: column;
                gap: 10px;
            }
            
            #editor {
                height: 300px;
            }
            
            .ql-editor {
                min-height: 250px;
                padding: 15px;
            }
            
            .image-preview {
                height: 120px;
            }
        }
        
        @media (max-width: 480px) {
            .admin-container {
                padding: 10px;
            }
            
            .form-container {
                padding: 15px;
            }
            
            #editor {
                height: 250px;
            }
            
            .ql-editor {
                min-height: 200px;
                padding: 12px;
            }
            
            .btn {
                padding: 12px 15px;
                font-size: 13px;
            }
        }
        
        /* Word Count */
        .word-count {
            font-size: 13px;
            color: #666;
            text-align: right;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Header -->
        <div class="admin-header">
            <h1>📝 Create New Post</h1>
            <div class="header-actions">
                <a href="posts.php" class="btn btn-light">
                    ← Back to Posts
                </a>
                <a href="index.php" class="btn btn-light">
                    🏠 Dashboard
                </a>
            </div>
        </div>
        
        <!-- Error Messages -->
        <?php if (!empty($errors)): ?>
            <ul class="errors">
                <?php foreach ($errors as $error): ?>
                    <li><?= htmlspecialchars($error) ?></li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>
        
        <!-- Create Form -->
        <form method="POST" action="" enctype="multipart/form-data" class="form-container" id="postForm">
            <!-- Title -->
            <div class="form-group">
                <label for="title" class="required">Post Title</label>
                <input type="text" 
                       id="title" 
                       name="title" 
                       class="form-control" 
                       value="<?= htmlspecialchars($_POST['title'] ?? '') ?>" 
                       required
                       maxlength="200"
                       placeholder="Enter your post title..."
                       autofocus>
                <div class="char-counter">
                    <span id="title-counter">0</span>/200 characters
                </div>
            </div>
            
            <!-- Content -->
            <div class="form-group">
                <label for="content" class="required">Post Content</label>
                <div class="editor-wrapper">
                    <div id="editor">
                        <?= htmlspecialchars($_POST['content'] ?? '') ?>
                    </div>
                    <textarea id="content" name="content" style="display: none;"></textarea>
                </div>
                <div class="word-count" id="word-count">Words: 0</div>
            </div>
            
            <!-- Excerpt -->
            <div class="form-group">
                <label for="excerpt">Short Excerpt (Optional)</label>
                <textarea id="excerpt" 
                          name="excerpt" 
                          class="form-control" 
                          rows="3"
                          placeholder="Brief summary of your post..."><?= htmlspecialchars($_POST['excerpt'] ?? '') ?></textarea>
                <div class="char-counter">
                    <span id="excerpt-counter">0</span>/500 characters
                </div>
            </div>
            
            <!-- Featured Image -->
            <div class="form-group">
                <label for="featured_image">Featured Image</label>
                <div class="image-upload">
                    <div class="no-image" id="no-image">
                        No image selected
                    </div>
                    <img id="image-preview" class="image-preview" alt="Image preview">
                    
                    <div class="file-input-container">
                        <input type="file" 
                               id="featured_image" 
                               name="featured_image" 
                               class="file-input" 
                               accept="image/*"
                               onchange="previewImage(this)">
                        <label for="featured_image" class="file-input-label">
                            📷 Choose Image
                        </label>
                    </div>
                    
                    <div class="file-info">
                        JPG, PNG, GIF, or WebP • Max 5MB
                    </div>
                </div>
            </div>
            
            <!-- Status -->
            <div class="form-group">
                <label>Status</label>
                <div class="status-options">
                    <label class="status-option active" onclick="setStatus('draft')">
                        <input type="radio" name="status" value="draft" checked>
                        💾 Save as Draft
                    </label>
                    <label class="status-option" onclick="setStatus('published')">
                        <input type="radio" name="status" value="published">
                        🚀 Publish Now
                    </label>
                </div>
            </div>
            
            <!-- Form Actions -->
            <div class="form-actions">
                <button type="button" class="btn btn-light" onclick="clearForm()">
                    Clear
                </button>
                <button type="submit" class="btn btn-secondary" name="action" value="draft">
                    Save Draft
                </button>
                <button type="submit" class="btn btn-primary" name="action" value="publish">
                    Publish Post
                </button>
            </div>
        </form>
    </div>
    
    <!-- Quill Editor Configuration -->
    <script>
        // Initialize Quill editor
        const quill = new Quill('#editor', {
            theme: 'snow',
            modules: {
                toolbar: [
                    // Text formatting
                    ['bold', 'italic', 'underline', 'strike'],
                    // Headers
                    [{ 'header': [1, 2, 3, false] }],
                    // Lists
                    [{ 'list': 'ordered'}, { 'list': 'bullet' }],
                    // Alignment
                    [{ 'align': [] }],
                    // Colors
                    [{ 'color': [] }, { 'background': [] }],
                    // Links and images
                    ['link', 'image'],
                    // Clean formatting
                    ['clean']
                ]
            },
            placeholder: 'Start writing your post here...'
        });
        
        // Update hidden textarea with editor content
        quill.on('text-change', function() {
            document.getElementById('content').value = quill.root.innerHTML;
            updateWordCount();
        });
        
        // Set initial content
        const initialContent = `<?= addslashes($_POST['content'] ?? '') ?>`;
        if (initialContent) {
            quill.root.innerHTML = initialContent;
            document.getElementById('content').value = initialContent;
        }
        
        // Character counters
        const titleInput = document.getElementById('title');
        const excerptInput = document.getElementById('excerpt');
        
        const titleCounter = document.getElementById('title-counter');
        const excerptCounter = document.getElementById('excerpt-counter');
        
        function updateCounters() {
            titleCounter.textContent = titleInput.value.length;
            excerptCounter.textContent = excerptInput.value.length;
        }
        
        titleInput.addEventListener('input', updateCounters);
        excerptInput.addEventListener('input', updateCounters);
        updateCounters(); // Initial count
        
        // Word count
        function updateWordCount() {
            const text = quill.getText().trim();
            const words = text ? text.split(/\s+/).length : 0;
            document.getElementById('word-count').textContent = `Words: ${words}`;
        }
        
        updateWordCount(); // Initial count
        
        // Image preview
        function previewImage(input) {
            const preview = document.getElementById('image-preview');
            const noImage = document.getElementById('no-image');
            
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    preview.src = e.target.result;
                    preview.style.display = 'block';
                    noImage.style.display = 'none';
                }
                
                reader.readAsDataURL(input.files[0]);
            }
        }
        
        // Status selection
        function setStatus(status) {
            const options = document.querySelectorAll('.status-option');
            options.forEach(option => {
                option.classList.remove('active');
                const radio = option.querySelector('input[type="radio"]');
                radio.checked = radio.value === status;
                if (radio.value === status) {
                    option.classList.add('active');
                }
            });
        }
        
        // Clear form
        function clearForm() {
            if (confirm('Are you sure you want to clear the form?')) {
                document.getElementById('postForm').reset();
                quill.setText('');
                document.getElementById('image-preview').style.display = 'none';
                document.getElementById('no-image').style.display = 'flex';
                updateCounters();
                updateWordCount();
                setStatus('draft');
            }
        }
        
        // Form submission handling
        const form = document.getElementById('postForm');
        form.addEventListener('submit', function(e) {
            // Update status based on button clicked
            const submitButton = e.submitter;
            if (submitButton && submitButton.name === 'action') {
                const status = submitButton.value;
                setStatus(status);
            }
            
            // Ensure content is updated
            document.getElementById('content').value = quill.root.innerHTML;
            
            // Show loading state
            if (submitButton) {
                const originalText = submitButton.innerHTML;
                submitButton.innerHTML = '<span class="loading"></span> Saving...';
                submitButton.disabled = true;
                
                setTimeout(() => {
                    submitButton.innerHTML = originalText;
                    submitButton.disabled = false;
                }, 3000);
            }
        });
        
        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Ctrl/Cmd + B for bold
            if ((e.ctrlKey || e.metaKey) && e.key === 'b') {
                e.preventDefault();
                quill.format('bold', !quill.getFormat().bold);
            }
            
            // Ctrl/Cmd + I for italic
            if ((e.ctrlKey || e.metaKey) && e.key === 'i') {
                e.preventDefault();
                quill.format('italic', !quill.getFormat().italic);
            }
            
            // Ctrl/Cmd + K for link
            if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
                e.preventDefault();
                const range = quill.getSelection();
                if (range && range.length > 0) {
                    const url = prompt('Enter link URL:', 'https://');
                    if (url) {
                        quill.format('link', url);
                    }
                }
            }
            
            // Ctrl/Cmd + S to save draft
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                e.preventDefault();
                document.querySelector('button[value="draft"]').click();
            }
            
            // Ctrl/Cmd + Enter to publish
            if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
                e.preventDefault();
                document.querySelector('button[value="publish"]').click();
            }
        });
        
        // Mobile toolbar optimization
        function optimizeForMobile() {
            if (window.innerWidth <= 768) {
                // Simplify toolbar for mobile
                const toolbar = document.querySelector('.ql-toolbar');
                if (toolbar) {
                    // Hide some buttons on mobile to save space
                    const buttons = toolbar.querySelectorAll('.ql-formats');
                    if (buttons.length > 3) {
                        for (let i = 3; i < buttons.length; i++) {
                            buttons[i].style.display = 'none';
                        }
                    }
                }
            }
        }
        
        // Initialize on load
        document.addEventListener('DOMContentLoaded', function() {
            optimizeForMobile();
            updateCounters();
            updateWordCount();
        });
        
        // Re-optimize on resize
        window.addEventListener('resize', optimizeForMobile);
    </script>
</body>
</html>