<?php
session_start();
// Security headers
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

require_once __DIR__ . "/../includes/auth.php";
require_login();
require_once __DIR__ . "/../includes/config.php";

try {
    // Fetch stats in one query
    $stats = $pdo->query("
        SELECT 
            COUNT(*) as total_posts,
            SUM(CASE WHEN status='published' THEN 1 ELSE 0 END) as published_posts,
            SUM(CASE WHEN status='draft' THEN 1 ELSE 0 END) as draft_posts,
            COALESCE(SUM(views), 0) as total_views
        FROM blog_posts
    ")->fetch(PDO::FETCH_ASSOC);
    
    $total_posts = $stats['total_posts'];
    $published_posts = $stats['published_posts'];
    $draft_posts = $stats['draft_posts'];
    $total_views = $stats['total_views'];
    
} catch (PDOException $e) {
    error_log("Dashboard error: " . $e->getMessage());
    die("An error occurred while loading dashboard statistics.");
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Blog Management</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .admin-wrapper {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 15px;
            padding: 20px 30px;
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .header h1 {
            color: #333;
            font-size: 24px;
            font-weight: 600;
        }
        
        .header span {
            color: #0073aa;
        }
        
        .logout-btn {
            background: linear-gradient(to right, #ff416c, #ff4b2b);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 50px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            transition: transform 0.3s ease;
        }
        
        .logout-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(255, 65, 108, 0.4);
        }
        
        .dashboard-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }
        
        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 25px;
            text-align: center;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.15);
        }
        
        .stat-card h3 {
            color: #666;
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-bottom: 10px;
        }
        
        .stat-card .number {
            font-size: 36px;
            font-weight: 700;
            color: #0073aa;
            line-height: 1;
        }
        
        .stat-card.published .number { color: #28a745; }
        .stat-card.draft .number { color: #ffc107; }
        .stat-card.views .number { color: #17a2b8; }
        
        .quick-actions {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
        }
        
        .quick-actions h2 {
            color: #333;
            margin-bottom: 25px;
            font-size: 20px;
        }
        
        .action-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .action-btn {
            display: flex;
            align-items: center;
            gap: 12px;
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 20px;
            text-decoration: none;
            color: #333;
            transition: all 0.3s ease;
        }
        
        .action-btn:hover {
            background: #0073aa;
            color: white;
            border-color: #0073aa;
            transform: translateY(-3px);
        }
        
        .action-btn .icon {
            font-size: 24px;
        }
        
        .action-btn .text h4 {
            font-size: 16px;
            margin-bottom: 5px;
        }
        
        .action-btn .text p {
            font-size: 12px;
            color: #666;
        }
        
        .action-btn:hover .text p {
            color: rgba(255, 255, 255, 0.8);
        }
        
        @media (max-width: 768px) {
            .header {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            
            .dashboard-grid {
                grid-template-columns: 1fr;
            }
            
            .action-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="admin-wrapper">
        <div class="header">
            <h1>Welcome back, <span><?= htmlspecialchars($_SESSION['admin_username']) ?></span></h1>
            <a href="logout.php" class="logout-btn">🚪 Logout</a>
        </div>
        
        <div class="dashboard-grid">
            <div class="stat-card">
                <h3>Total Posts</h3>
                <div class="number"><?= $total_posts ?></div>
            </div>
            
            <div class="stat-card published">
                <h3>Published Posts</h3>
                <div class="number"><?= $published_posts ?></div>
            </div>
            
            <div class="stat-card draft">
                <h3>Draft Posts</h3>
                <div class="number"><?= $draft_posts ?></div>
            </div>
            
            <div class="stat-card views">
                <h3>Total Views</h3>
                <div class="number"><?= $total_views ?></div>
            </div>
        </div>
        
        <div class="quick-actions">
            <h2>Quick Actions</h2>
            <div class="action-grid">
                <a href="posts.php" class="action-btn">
                    <div class="icon">📄</div>
                    <div class="text">
                        <h4>Manage Posts</h4>
                        <p>View, edit, and organize all blog posts</p>
                    </div>
                </a>
                
                <a href="new-post.php" class="action-btn">
                    <div class="icon">➕</div>
                    <div class="text">
                        <h4>Create New Post</h4>
                        <p>Write and publish a new blog article</p>
                    </div>
                </a>
                
                <a href="profiles.php"  class="action-btn">
                    <div class="icon">🖼️</div>
                    <div class="text">
                        <h4>Manage Profile</h4>
                        <p>Update your email and password</p>
                    </div>
                </a>
                
                <a href="posts.php?filter=draft" class="action-btn">
                    <div class="icon">📝</div>
                    <div class="text">
                        <h4>View Drafts</h4>
                        <p>Continue editing unpublished posts</p>
                    </div>
                </a>
            </div>
        </div>
    </div>
</body>
</html>